/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package viewer;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.IOException;
import java.net.URISyntaxException;

import javax.swing.AbstractAction;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.KeyStroke;

import com.inet.report.EngineRenderData;
import com.inet.viewer.NavigationView;
import com.inet.viewer.RenderData;
import com.inet.viewer.ReportView;
import com.inet.viewer.SwingReportViewer;

import viewer.bookmark.BookmarkView;

/**
 * This is an example of how you can add your own custom sidebar to the viewer. Here we create a "BookmarkView" and add
 * it to the viewer via the viewer's navigation view. As well, we use the keyboard shortcut "Ctrl+D" to add bookmarks to
 * the view.
 * @see BookmarkView
 */
public class ViewerWithBookmarks {

    private BookmarkView            bookmarkView;

    private final SwingReportViewer viewer;

    private final ReportView        view;

    private AbstractAction          addBookmarkAction = new AbstractAction() {
                                                          @Override
                                                          public void actionPerformed( ActionEvent e ) {
                                                              // If Ctrl+D was pressed, we add a bookmark to the bookmark view
                                                              // (if this page was already bookmarked, this will do nothing)
                                                              int currPage = view.getCurrentPage();
                                                              bookmarkView.addBookmark( "Page #" + currPage, currPage );
                                                          }
                                                      };

    /**
     * Add bookmarks to the viewer.
     */
    public ViewerWithBookmarks() {

        // Here we create our Viewer instance
        viewer = new SwingReportViewer();

        // creating our keyboard shortcut
        InputMap localMap = viewer.getInputMap( JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT );
        KeyStroke keystrokeAddbookmark = KeyStroke.getKeyStroke( KeyEvent.VK_D, Toolkit.getDefaultToolkit().getMenuShortcutKeyMaskEx() );
        localMap.put( keystrokeAddbookmark, "addBookmark" );
        viewer.getActionMap().put( "addBookmark", addBookmarkAction );

        String reportLocation = "";
        try {
            reportLocation = getClass().getResource( "../sample.rpt" ).toURI().toString();
        } catch( URISyntaxException e ) {
            // Nothing to do
        }

        // then initialize the render data connection.
        RenderData data = new EngineRenderData( "report=" + reportLocation );

        // you will most likely have a report server already, so you can use the URL: http://server:port/report.rpt
        // RenderData data = new URLRenderData( "http://server:port/?report=file:c:/report1.rpt" );

        view = viewer.addNewReportView( data );
        bookmarkView = new BookmarkView( view );

        NavigationView navView = view.getNavigationView();

        // Adding the bookmark view
        navView.addNavigationTab( BookmarkView.TITLE, bookmarkView );
        // Making sure the navigation view is shown.
        navView.setVisible( true );
    }

    /**
     * Initialize the GUI.
     */
    public static void initGUI() {
        ViewerWithBookmarks myApp = new ViewerWithBookmarks();
        // Note that the new Viewer cannot be added to an AWT Frame.
        JFrame frame = new JFrame( "i-net Clear Reports Viewer" );
        frame.setDefaultCloseOperation( JFrame.EXIT_ON_CLOSE );
        frame.getContentPane().add( myApp.viewer );

        //Display the window.
        frame.pack();
        frame.setVisible( true );
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     * @throws IOException in case of IO errors. Often the port is already in use.
     */
    public static void main( String[] args ) throws IOException {
        initGUI();
    }
}
